const mysql = require('mysql2/promise');
const fs = require('fs');
const path = require('path');

// Load environment variables
require('dotenv').config();

// MySQL configuration from environment variables
const dbConfig = {
  host: process.env.DB_HOST,
  port: parseInt(process.env.DB_PORT) || 3306,
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};

const SQL_INIT_PATH = path.join(__dirname, 'init.sql');

// Connection pool for better performance
let connectionPool = null;

function initDatabase() {
  return new Promise(async (resolve, reject) => {
    try {
      // Create connection pool
      connectionPool = mysql.createPool(dbConfig);
      
      console.log('Connected to MySQL database');

      // Read and execute initialization SQL
      const sql = fs.readFileSync(SQL_INIT_PATH, 'utf8');
      
      // Split SQL statements and execute them one by one
      const statements = sql.split(';').filter(stmt => stmt.trim() !== '');
      
      const connection = await connectionPool.getConnection();
      
      try {
        for (const statement of statements) {
          if (statement.trim()) {
            await connection.execute(statement);
          }
        }
        
        console.log('Database initialized successfully');
        resolve();
      } catch (err) {
        console.error('Error executing init.sql:', err);
        reject(err);
      } finally {
        connection.release();
      }
    } catch (err) {
      console.error('Error connecting to database:', err);
      reject(err);
    }
  });
}

// Enhanced function to get database connection from pool
function getDb() {
  if (!connectionPool) {
    throw new Error('Database pool not initialized. Call initDatabase() first.');
  }
  return connectionPool;
}

// Return connection to pool (not needed with mysql2 pool, but kept for compatibility)
function returnToPool(connection) {
  // With mysql2 pool, connections are automatically returned when released
  // This function is kept for backward compatibility
  if (connection && connection.release) {
    connection.release();
  }
}

// Clean up pool on process exit
process.on('SIGINT', async () => {
  console.log('Cleaning up database connections...');
  if (connectionPool) {
    await connectionPool.end();
  }
  process.exit(0);
});

process.on('SIGTERM', async () => {
  console.log('Cleaning up database connections...');
  if (connectionPool) {
    await connectionPool.end();
  }
  process.exit(0);
});
// Run initialization if this file is executed directly
if (require.main === module) {
  initDatabase()
    .then(async () => {
      console.log('Database setup completed');
      if (connectionPool) {
        await connectionPool.end();
      }
      process.exit(0);
    })
    .catch(async (error) => {
      console.error('Database setup failed:', error);
      if (connectionPool) {
        await connectionPool.end();
      }
      process.exit(1);
    });
}

module.exports = { initDatabase, getDb, returnToPool };
